//*************************************************************************************************
//
//	Description:
//		simplebatch.fx - simple batch of triangles shader, optionally textured.
//
//	<P> Copyright (c) 2009 Slightly Mad Studios Ltd. All rights reserved.
//
//	Author: 
//		TNettleship


//
// Textures
//

#include "stddefs.fxh"


//
// Transforms
//
SHARE_PARAM float4x4 projMatrix : projection
<
	bool appEdit = false;
	bool export = false;
>;


texture BaseTexture : TEXTURE
<
	bool appEdit = true;
>;

bool displayIn2D
<
	bool appEdit = true;
> = false;



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D BaseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="BaseTexture"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 6;
> 
= sampler_state
{
	Texture = < BaseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;

#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
#endif
	SET_MAX_ANISOTROPY( 6 )
#endif
};


//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structures
struct VSINPUT_UNTEXTURED
{
	float3 position : POSITION;
	float4 colour : COLOR0;
};

struct VSINPUT_TEXTURED
{
	float3 position : POSITION;
	float4 colour : COLOR0;
	float2 texcoord : TEXCOORD0;
};


// Output structures (also input to the fragment shader)
struct VSOUTPUT_UNTEXTURED
{
	float4 position		: POSITION;							// View-coords position
	float4 colour : TEXCOORD0;								// Vertex colour
};

struct VSOUTPUT_TEXTURED
{
	float4 position		: POSITION;							// View-coords position
	float4 colour : TEXCOORD0;				// Vertex colour
	float2 texcoord : TEXCOORD1;		// Texture coord
};

struct PSINPUT_UNTEXTURED
{
	float4 position		: POSITION;							// View-coords position
	float4 colour : TEXCOORD0_centroid;								// Vertex colour
};

struct PSINPUT_TEXTURED
{
	float4 position		: POSITION;							// View-coords position
	float4 colour : TEXCOORD0_centroid;				// Vertex colour
	float2 texcoord : TEXCOORD1_centroid;		// Texture coord
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT_UNTEXTURED SimpleBatchVertexShader( VSINPUT_UNTEXTURED _input )
{
	VSOUTPUT_UNTEXTURED _output;

	if ( displayIn2D )
	{
		_output.position = float4( _input.position, 1.0f );
	}
	else
	{
		_output.position = mul( float4( _input.position, 1.0f ), projMatrix );
	}
	_output.colour = _input.colour;
	

	return _output;
}

VSOUTPUT_TEXTURED SimpleBatchTexturedVertexShader( VSINPUT_TEXTURED _input )
{
	VSOUTPUT_TEXTURED _output;

	if ( displayIn2D )
	{
		_output.position = float4( _input.position, 1.0f );
	}
	else
	{
		_output.position = mul( float4( _input.position, 1.0f ), projMatrix );
	}

	_output.colour = _input.colour;
	_output.texcoord = _input.texcoord;

	return _output;
}


//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT SimpleBatchFragmentShader( PSINPUT_UNTEXTURED _input )
{
	PSOUTPUT _output;

	_output.Colour = _input.colour;

	return _output;
}


PSOUTPUT SimpleBatchTexturedFragmentShader( PSINPUT_TEXTURED _input )
{
	PSOUTPUT _output;

	_output.Colour = tex2D( BaseMap, _input.texcoord ) * _input.colour;

	return _output;
}



PSOUTPUT SimpleBatchPremultipliedAlphaFragmentShader( PSINPUT_UNTEXTURED _input )
{
	PSOUTPUT _output;

	float4 colour = _input.colour;

	// Premultiply colour by alpha
	colour.rgb *= colour.a;

	_output.Colour = colour;

	return _output;
}


PSOUTPUT SimpleBatchPremultipliedAlphaTexturedFragmentShader( PSINPUT_TEXTURED _input )
{
	PSOUTPUT _output;

	float4 colour = tex2D( BaseMap, _input.texcoord ) * _input.colour;

	// Premultiply colour by alpha
	colour.rgb *= colour.a;

	_output.Colour = colour;

	return _output;
}


//-----------------------------------------------------------------------
//
// Technique(s)
//

technique SimpleBatch
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = true;

	string normalBehaviour	= "ERMB_RENDER";
	string normalTechnique	= "SimpleBatch";
	int    normalDeferredID	= 0;
>
{
	pass Pass0
	{
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx SimpleBatchVertexShader();
		PixelShader = compile sce_fp_rsx SimpleBatchFragmentShader();
#else
		VertexShader = compile vs_3_0 SimpleBatchVertexShader();
		PixelShader = compile ps_3_0 SimpleBatchFragmentShader();
#endif
	}
}



technique SimpleBatchTextured
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = true;

	string normalBehaviour	= "ERMB_RENDER";
	string normalTechnique	= "SimpleBatchTextured";
	int    normalDeferredID	= 0;
>
{
	pass Pass0
	{
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx SimpleBatchTexturedVertexShader();
		PixelShader = compile sce_fp_rsx SimpleBatchTexturedFragmentShader();
#else
		VertexShader = compile vs_3_0 SimpleBatchTexturedVertexShader();
		PixelShader = compile ps_3_0 SimpleBatchTexturedFragmentShader();
#endif
	}
}



technique SimpleBatch_PremultipliedAlpha
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = true;

	string normalBehaviour	= "ERMB_RENDER";
	string normalTechnique	= "SimpleBatch_PremultipliedAlpha";
	int    normalDeferredID	= 0;
>
{
	pass Pass0
	{
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx SimpleBatchVertexShader();
		PixelShader = compile sce_fp_rsx SimpleBatchPremultipliedAlphaFragmentShader();
#else
		VertexShader = compile vs_3_0 SimpleBatchVertexShader();
		PixelShader = compile ps_3_0 SimpleBatchPremultipliedAlphaFragmentShader();
#endif
	}
}


technique SimpleBatchTextured_PremultipliedAlpha
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = true;

	string normalBehaviour	= "ERMB_RENDER";
	string normalTechnique	= "SimpleBatchTextured_PremultipliedAlpha";
	int    normalDeferredID	= 0;
>
{
	pass Pass0
	{
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx SimpleBatchTexturedVertexShader();
		PixelShader = compile sce_fp_rsx SimpleBatchPremultipliedAlphaTexturedFragmentShader();
#else
		VertexShader = compile vs_3_0 SimpleBatchTexturedVertexShader();
		PixelShader = compile ps_3_0 SimpleBatchPremultipliedAlphaTexturedFragmentShader();
#endif
	}
}
